import os
import psutil
import time
import threading
from flask import Flask, jsonify

app = Flask(__name__)

# 全局变量存储最新网速和锁，用于确保多线程环境下对共享数据的安全访问
# Global variable to store the latest network speed and a lock
# to ensure safe access to shared data in a multi-threaded environment.
current_speed = {"dSpeed": 0.0, "uSpeed": 0.0}
speed_lock = threading.Lock() # 用于保护 current_speed 的锁 # Lock to protect current_speed

# 获取 CPU 使用率
# Get CPU usage percentage
def get_cpu_usage():
    # interval=1 表示在过去1秒内的CPU平均使用率
    # interval=1 means the average CPU usage over the past 1 second.
    return psutil.cpu_percent(interval=1)

# 获取内存使用情况百分比
# Get memory usage percentage
def get_memory_usage():
    # psutil.virtual_memory().percent 直接返回内存使用率百分比，无需手动计算
    # psutil.virtual_memory().percent directly returns the memory usage percentage,
    # no manual calculation needed.
    return psutil.virtual_memory().percent

# 获取 CPU 温度
# Get CPU temperature
def get_cpu_temperature():
    try:
        # 尝试从标准路径读取CPU温度
        # Try to read CPU temperature from a standard path
        with open("/sys/class/thermal/thermal_zone0/temp", "r") as f:
            temp = f.read().strip()
        return float(temp) / 1000  # 温度单位转换为摄氏度 # Convert temperature unit to Celsius
    except FileNotFoundError:
        # 如果文件不存在，表示可能不支持或路径不同
        # If the file does not exist, it means it might not be supported or the path is different.
        print("Warning: CPU temperature file not found. This feature might not be supported on this system.")
        return None
    except Exception as e:
        # 捕获其他可能的错误
        # Catch other possible errors
        print(f"Error getting CPU temperature: {e}")
        return None

# 后台线程：每秒更新网速
# Background thread: Update network speed every second
def update_network_speed():
    """后台线程：每秒更新网速"""
    global current_speed, speed_lock
    while True:
        # 记录开始时的网络I/O计数器
        # Record network I/O counters at the start
        net_io_start = psutil.net_io_counters()
        prev_rx = net_io_start.bytes_recv
        prev_tx = net_io_start.bytes_sent
        start_time = time.time()

        # 等待1秒
        # Wait for 1 second
        time.sleep(1)

        # 记录结束时的网络I/O计数器
        # Record network I/O counters at the end
        net_io_end = psutil.net_io_counters()
        current_rx = net_io_end.bytes_recv
        current_tx = net_io_end.bytes_sent
        elapsed = time.time() - start_time # 实际经过的时间 # Actual elapsed time

        # 计算下载和上传速度 (Bytes/second)
        # Calculate download and upload speed (Bytes/second)
        download_speed_bps = (current_rx - prev_rx) / elapsed
        upload_speed_bps = (current_tx - prev_tx) / elapsed

        # 转换为 KB/s 并保留两位小数
        # Convert to KB/s and round to 2 decimal places
        # 注意：1 KB = 1024 Bytes
        # Note: 1 KB = 1024 Bytes
        d_speed_kbps = round(download_speed_bps / 1024, 2)
        u_speed_kbps = round(upload_speed_bps / 1024, 2)

        # 使用锁来安全地更新全局变量
        # Use a lock to safely update the global variable
        with speed_lock:
            current_speed["dSpeed"] = d_speed_kbps
            current_speed["uSpeed"] = u_speed_kbps

# 启动后台线程，设置为守护线程，确保主程序退出时它也退出
# Start the background thread, set as daemon to ensure it exits when the main program exits.
network_thread = threading.Thread(target=update_network_speed, daemon=True)
network_thread.start()

@app.route("/system_stats", methods=["GET"])
def system_stats():
    # 获取系统信息
    # Get system information
    with speed_lock:
        # 在读取全局变量时也要使用锁
        # Use the lock when reading the global variable as well
        stats = {
            "cpuPer": get_cpu_usage(),
            "cpuTemp": get_cpu_temperature(),
            "ramPer": get_memory_usage(),
            "dSpeed": current_speed["dSpeed"], # KB/s
            "uSpeed": current_speed["uSpeed"]  # KB/s
        }

    # 返回 JSON 格式的数据
    # Return data in JSON format
    return jsonify(stats)

if __name__ == "__main__":
    # 在所有网络接口上监听 5000 端口
    # Listen on port 5000 on all network interfaces
    # 注意：debug=True 仅用于开发环境，生产环境应设置为 False
    # Note: debug=True is for development environment only, should be False in production.
    app.run(host="0.0.0.0", port=5000, debug=True)